/*
**************************************************************************
*                                                                         *
*                             COPYRIGHT NOTICE                            *
*                                                                         *
* This software/database is categorized as "United States Government      *
* Work" under the terms of the United States Copyright Act.  It was       *
* produced as part of the author's official duties as a Government        *
* employee and thus can not be copyrighted.  This software/database is    *
* freely available to the public for use without a copyright notice.      *
* Restrictions can not be placed on its present or future use.            *
*                                                                         *
* Although all reasonable efforts have been taken to ensure the accuracy  *
* and reliability of the software and data, the National Library of       *
* Medicine (NLM) and the U.S. Government do not and can not warrant the   *
* performance or results that may be obtained by using this software,     *
* data, or derivative works thereof.  The NLM and the U.S. Government     *
* disclaim any and all warranties, expressed or implied, as to the        *
* performance, merchantability or fitness for any particular purpose or   *
* use.                                                                    *
*                                                                         *
* In any work or product derived from this material, proper attribution   *
* of the author(s) as the source of the software or data would be         *
* appreciated.                                                            *
*                                                                         *
************************************************************************** 

	Tom Madden (12/28/2001)
*/

#include <ncbi.h>
#include <salpacc.h>
#include <blast.h>

#define BUFFER_LEN 50

/*
	Print a report on hits with start/stop.  Zero-offset is used. 
*/
static void MySeqAlignPrint(SeqAlignPtr seqalign, FILE *outfp)
{
     Char query_id_buf[BUFFER_LEN+1], target_id_buf[BUFFER_LEN+1];
     SeqIdPtr query_id, target_id;

     while (seqalign)
     {
	query_id = SeqAlignId(seqalign, 0);
     	SeqIdWrite(query_id, query_id_buf, PRINTID_FASTA_LONG, BUFFER_LEN);

	target_id = SeqAlignId(seqalign, 1);
	SeqIdWrite(target_id, target_id_buf, PRINTID_FASTA_LONG, BUFFER_LEN);
	
	fprintf(outfp, "%s:%ld-%-20ld\t%s:%ld-%ld\n", 
	   query_id_buf, (long) SeqAlignStart(seqalign, 0), (long) SeqAlignStop(seqalign, 0),
	   target_id_buf, (long) SeqAlignStart(seqalign, 1), (long) SeqAlignStop(seqalign, 1));

	seqalign = seqalign->next;
     }

     return;
}

#define NUMARG (sizeof(myargs)/sizeof(myargs[0]))

static Args myargs [] = {
    { "Program Name",           /* 0 */
      NULL, NULL, NULL, FALSE, 'p', ARG_STRING, 0.0, 0, NULL},
    { "Database",               /* 1 */
      "nr", NULL, NULL, FALSE, 'd', ARG_STRING, 0.0, 0, NULL},
    { "Query File",             /* 2 */
      "stdin", NULL, NULL, FALSE, 'i', ARG_FILE_IN, 0.0, 0, NULL},
    { "Expectation value (E)",  /* 3 */
      "10.0", NULL, NULL, FALSE, 'e', ARG_FLOAT, 0.0, 0, NULL},
    { "Output file",             /* 4 */
      "stdout", NULL, NULL, FALSE, 'o', ARG_FILE_OUT, 0.0, 0, NULL},
};

Int2 Main (void)
 
{
    BioseqPtr query_bsp;
    BLAST_OptionsBlkPtr options;
    Boolean db_is_na, query_is_na;
    SeqAlignPtr  seqalign;
    SeqEntryPtr sep;
    CharPtr blast_program, blast_database, blast_inputfile, blast_outputfile;
    FILE *infp, *outfp;
    
    if (! GetArgs ("doblast", NUMARG, myargs)) {
        return 1;
    }
    
    UseLocalAsnloadDataAndErrMsg ();
    
    if (! SeqEntryLoad())
        return 1;
    
    ErrSetMessageLevel(SEV_WARNING);
    
    blast_program = myargs [0].strvalue;
    blast_database = myargs [1].strvalue;
    blast_inputfile = myargs [2].strvalue;
    blast_outputfile = myargs [4].strvalue;
    
    if ((infp = FileOpen(blast_inputfile, "r")) == NULL) {
        ErrPostEx(SEV_FATAL, 0, 0, "blast: Unable to open input file %s\n", blast_inputfile);
        return 2;
    }

    if ((outfp = FileOpen(blast_outputfile, "r")) == NULL) {
        ErrPostEx(SEV_FATAL, 0, 0, "blast: Unable to open output file %s\n", blast_outputfile);
        return 2;
    }

    BlastGetTypes(blast_program, &query_is_na, &db_is_na);
    
    sep = FastaToSeqEntry(infp, query_is_na);
    if (sep == NULL) {
        ErrPostEx(SEV_FATAL, 0, 0, "blast: Unable to obtain SeqEntry");
        return 3;
    }
          
    query_bsp = NULL;
    if (query_is_na) {
            SeqEntryExplore(sep, &query_bsp, FindNuc);
    } else {
            SeqEntryExplore(sep, &query_bsp, FindProt);
    }
          
    if (query_bsp == NULL) {
             ErrPostEx(SEV_FATAL, 0, 0, "Unable to obtain bioseq\n");
             return 4;
    }

    /* Get default options. */
    options = BLASTOptionNew(blast_program, TRUE);
    if (options == NULL)
        return 5;

    options->expect_value  = (Nlm_FloatHi) myargs [3].floatvalue;
        
    /* Perform the actual search. */
    seqalign = BioseqBlastEngine(query_bsp, blast_program, blast_database, options, 
	NULL, NULL, NULL);

    /* Do something with the SeqAlign... */
    MySeqAlignPrint(seqalign, outfp);
    
    /* clean up. */
    seqalign = SeqAlignSetFree(seqalign);
    options = BLASTOptionDelete(options);

    sep = SeqEntryFree(sep);
    FileClose(infp);
    FileClose(outfp);
    
    return 0;
}
