
/*
 * 
 *  file:  ./src/edu/virginia/bioch/nopt/alignments/Alignment.java
 * 
 *  Copyright (c) 2004,  the University of Virginia.
 *  All rights reverved.
 * 
 *  See the file COPYRIGHT in the top directory of this distribution for
 *  more information.
 *  
 *  THE SOFTWARE IS PROVIDED _AS IS_, WITHOUT WARRANTY OF ANY KIND, EXPRESS 
 *  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.  
 *  
 */ 

package edu.virginia.bioch.nopt.alignments;



import java.util.*;
import java.lang.*;
import java.awt.*;
import java.awt.event.*;
import java.io.*;                                                   
import javax.swing.*;
import javax.swing.event.*;

import edu.virginia.bioch.util.*;
import edu.virginia.bioch.nopt.alignments.*;
import edu.virginia.bioch.scoringmatrices.*;
import edu.virginia.bioch.nopt.options.*;
import edu.virginia.bioch.nopt.conditions.*;
import edu.virginia.bioch.nopt.alignments.*;


public class Alignment
{
	public static final char match = '=';
	public static final char botGap = '+';  // top insertion
	public static final char topGap = '-';  // bot insertion

	protected float _percentSimilarity;
	protected float _percentIdentity;
	protected Vector _edges;
	protected int _length;
	protected String _seq1;
	protected String _seq2;
	protected String _paramKey;
	protected int _score;
	protected int _begin1Index;
	protected int _begin2Index;
	protected int _end1Index;
	protected int _end2Index;
	protected AlignmentHandler ah;

	//
	// Constructor
	//
	public Alignment(String s, AlignmentHandler ah) 
		throws IllegalStateException, 
			   IOException,
			   IllegalAccessException, 
			   ClassNotFoundException, 
			   NoSuchFieldException, 
			   SecurityException
	{
		System.out.println("Alignment constructor");
		this.ah = ah;

		//
		//  Parse input string.
		//
		if ( !_parseInput(s) )
			throw( new IllegalStateException("couldn't parse string:\n" + s) );
	

		_constructor();
	}

	public Alignment( AlignmentHandler ah, String s1, String s2, String pk,
	                  int bi1, int bi2, int ei1, int ei2 )
	{
		this.ah = ah;	
		_seq1 = s1.toUpperCase();
		_seq2 = s2.toUpperCase();
		_paramKey = pk; 
		_begin1Index = bi1; 
		_begin2Index = bi2; 
		_end1Index = ei1; 
		_end2Index = ei2; 
		AlignmentParameters apx = getAlignmentParameters();
		_score = Alignment.calcScore( s1, s2, apx ); 
		
		_constructor();
	}

	private void _constructor()
	{
		_length = _seq1.length();

		// 
		// Calculate various stats.  Importantly the relativePositions.
		//
		AlignmentParameters ap = getAlignmentParameters();  
		ScoringMatrix sm = ap.getScoringMatrix();

		int simCount = 0;
		int matchCount = 0;

		_edges = new Vector();

		int x = _begin1Index; 
		int y = _begin2Index; 
		int prevX = _begin1Index;
		int prevY = _begin2Index;
		int dispOffset = Math.max(_begin1Index,_begin2Index);
		EdgeKey key = new EdgeKey();

		for ( int i = 0; i < _length; i++ )
		{
			char a = _seq1.charAt(i);
			char b = _seq2.charAt(i);

			if ( a != '-' && b != '-' ) 
			{
				if ( a == b )
					matchCount++;

				if ( sm.score( a, b ) > 0 )
					simCount++;

				x++;
				y++;
			}
			else if ( a == '-' && b != '-' )
			{
				y++;
			}
			else if ( a != '-' && b == '-' )
			{
				x++;
			}
	
			double relPos = ((double)(i+dispOffset))/
			                (double)(_length+dispOffset);
	
			key.set( prevX, prevY, x, y, a, b );

			ah.updateMaxLength( _length+dispOffset );

			_edges.add( ah.updateEdges( key, relPos ) );

			prevX = x;
			prevY = y;
		}

		_percentIdentity = (float)matchCount/(float)_length;
		_percentSimilarity = (float)simCount/(float)_length;

		ah.updatePercentIdentity( _percentIdentity );
		ah.updatePercentSimilarity( _percentSimilarity );

	} 

	private String createKey(int px, int py, int x, int y, String a, String b)
	{ 
		String s = Integer.toString(px) + "," + Integer.toString(py) + "|" +
		           Integer.toString(x)  + "," + Integer.toString(y) + "|" +
				   a + "," + b;
		return s;
	}

	public Vector getEdges() { return _edges; }

	//
	// Parses input string
	//
	private boolean _parseInput( String inputString ) 
		throws IOException, IllegalStateException
	{
		if ( inputString != null )
		{
			StringTokenizer t = new StringTokenizer(inputString,"|");
			_paramKey = t.nextToken();
			_begin1Index = Integer.parseInt( t.nextToken() );
			_begin2Index = Integer.parseInt( t.nextToken() );
			_end1Index = Integer.parseInt( t.nextToken() );
			_end2Index = Integer.parseInt( t.nextToken() );
			_score = Integer.parseInt( t.nextToken() );
			String compressedString = t.nextToken();
			uncompress( compressedString );
			return true;
		}
		else
			return false;
	}

	private void uncompress( String compressedString )
	{
		StringBuffer compressedBuffer = new StringBuffer( compressedString );

		AlignmentParameters ap = getAlignmentParameters();

		StringBuffer ps1 = new StringBuffer( ap.getSeq1() );
		StringBuffer ps2 = new StringBuffer( ap.getSeq2() );

		StringBuffer s1 = new StringBuffer();
		StringBuffer s2 = new StringBuffer();

		StringBuffer num = new StringBuffer();

		int s1Count = _begin1Index;
		int s2Count = _begin2Index;

		char sentinel = '0';  // shouldn't match anything

		int i = 0;
		while ( i < compressedBuffer.length() )
		{
			sentinel = compressedBuffer.charAt( i++ );

			while ( i < compressedBuffer.length() &&
			        compressedBuffer.charAt( i ) != match  &&
			        compressedBuffer.charAt( i ) != topGap &&
			        compressedBuffer.charAt( i ) != botGap )
				num.append( compressedBuffer.charAt( i++ ) );

			int count = Integer.parseInt( num.toString() );
			num.delete(0,num.length());

			if ( sentinel == match )
			{
				while ( count > 0 )
				{
					s1.append( ps1.charAt( s1Count++ ) ); 
					s2.append( ps2.charAt( s2Count++ ) ); 
					count--;
				}
			}
			else if ( sentinel == topGap )
			{
				while ( count > 0 )
				{
					s1.append( "-" );
					s2.append( ps2.charAt( s2Count++ ) ); 
					count--;
				}
			}
			else if ( sentinel == botGap )
			{
				while ( count > 0 )
				{
					s1.append( ps1.charAt( s1Count++ ) ); 
					s2.append( "-" );
					count--;
				}
			}
		}
		
		_seq1 = new String( s1 );
		_seq2 = new String( s2 );
	}


	//
	// Get methods
	//
	public String getSeq1() { return _seq1; }
	public String getSeq2() { return _seq2; }
	public int getScore() { return _score; }
	public float getPercentSimilarity() { return _percentSimilarity; }
	public float getPercentIdentity() { return _percentIdentity; }
	public int getLength() { return _length; }
	public int getBegin1Index() { return _begin1Index; }
	public int getBegin2Index() { return _begin2Index; }
	public int getEnd1Index() { return _end1Index; }
	public int getEnd2Index() { return _end2Index; }
	public String getParamKey() { return _paramKey; }

	public AlignmentParameters getAlignmentParameters()
	{
		AlignmentParameters ap = null;
		try {
			ap = ah.getAlignmentParameters( _paramKey );
		} catch (Exception e) { e.printStackTrace(); }

		return ap; 
	}

	public static int calcScore( String seq1, String seq2, 
	                             AlignmentParameters ap )
	{

		if ( seq1.length() != seq2.length() )
			System.out.println("seqs not the same length: " + 
			                   seq1.length() + "  " + seq2.length() + 
							   "  Bad things iminent!" );

		int totalScore = 0;
		int seq1GapCount = 0;
		int seq2GapCount = 0;
		char gap = '-'; 
		int gapCreate = ap.getGapCreate();
		int gapExtend = ap.getGapExtend();
		ScoringMatrix sm = ap.getScoringMatrix();

		for ( int i = 0; i < seq1.length(); i++ )
		{
			if ( seq1.charAt(i) != gap && seq2.charAt(i) != gap )
			{
				totalScore += sm.score( seq1.charAt(i), seq2.charAt(i) );

				if ( seq1GapCount > 0 )
					totalScore += gapCreate + ( gapExtend * seq1GapCount );
				
				if ( seq2GapCount > 0 )
					totalScore += gapCreate + ( gapExtend * seq2GapCount );

				seq1GapCount = 0;
				seq2GapCount = 0;
			}
			else if ( seq1.charAt(i) != gap && seq2.charAt(i) == gap )
			{
				seq2GapCount++;

				if ( seq1GapCount > 0 )
				{
					totalScore += gapCreate + ( gapExtend * seq1GapCount );
					seq1GapCount = 0;
				}
			}
			else if ( seq1.charAt(i) == gap && seq2.charAt(i) != gap )
			{
				seq1GapCount++;

				if ( seq2GapCount > 0 )
				{
					totalScore += gapCreate + ( gapExtend * seq2GapCount );
					seq2GapCount = 0;
				}
			}
			else
				System.out.println("Not counting double gap at " + 
				                   seq1.charAt(i) + "  and  " + 
								   seq2.charAt(i) );
		}

		// account for any gaps on the end
		if ( seq1GapCount > 0 )
			totalScore += gapCreate + ( gapExtend * seq1GapCount );
		
		if ( seq2GapCount > 0 )
			totalScore += gapCreate + ( gapExtend * seq2GapCount );

		return totalScore;
	}
}

