
/*
 * 
 *  file:  ./src/edu/virginia/bioch/nopt/alignments/AlignmentHandler.java
 * 
 *  Copyright (c) 2004,  the University of Virginia.
 *  All rights reverved.
 * 
 *  See the file COPYRIGHT in the top directory of this distribution for
 *  more information.
 *  
 *  THE SOFTWARE IS PROVIDED _AS IS_, WITHOUT WARRANTY OF ANY KIND, EXPRESS 
 *  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.  
 *  
 */ 

package edu.virginia.bioch.nopt.alignments;


import java.util.*;
import java.lang.*;
import java.awt.*;
import java.awt.event.*;
import java.io.*;                                                   
import javax.swing.*;
import javax.swing.event.*;

import edu.virginia.bioch.util.*;
import edu.virginia.bioch.nopt.alignments.*;
import edu.virginia.bioch.scoringmatrices.*;
import edu.virginia.bioch.nopt.options.*;
import edu.virginia.bioch.nopt.actions.*;
import edu.virginia.bioch.nopt.conditions.*;


public class AlignmentHandler
	implements EdgeContainer, Reinitializable
{

	protected int _maxLength;
	protected int _maxCount;
	protected HashMap _allEdges;
	protected HashMap _params;
	protected String _order; 
	protected int _curr; 
	protected Vector _a; 
	protected float _optimalPercentSimilarity; 
	protected float _optimalPercentIdentity; 
	protected HashMap _scoringMatrixMap; 
	protected Map _config; 
	protected Alignment _currentAlignment;
	protected AlignmentSetReader _asr;

	protected String _seq1;
	protected String _seq2;
	protected String _seq1Name;
	protected String _seq2Name;
	protected String _seq1Desc;
	protected String _seq2Desc;

	// Dummy constructor. 
	// Don't use this for anything.
	// (Unless you know what you're doing).
	public AlignmentHandler( )
	{
		init ();
	}
	private void init( )
	{
		_a = new Vector();
		_curr = 0;
		_maxCount = 0;
		_maxLength = 0;
		_allEdges = new HashMap();
		_params = new HashMap();
		_order = "";
		_optimalPercentSimilarity = 0; 
		_optimalPercentIdentity = 0; 
		_scoringMatrixMap = new HashMap();
		_order = ""; 
		_currentAlignment = null;
		_seq1 = "";
		_seq2 = "";
		_seq1Name = "";
		_seq2Name = "";
		_seq1Desc = "";
		_seq2Desc = "";
	}

	public AlignmentHandler( AlignmentSetReader ar )
		throws Exception
	{
		init();

		_asr = ar;

		try {

		// add parameters
		Vector params = _asr.getParameters();
		for ( int i = 0; i <  params.size(); i++ )
		{
			AlignmentParameters ap = new AlignmentParameters( 
												(String)params.get(i) );

			// set the seqs the first time through
			if ( _seq1.equals("") )
			{
				_seq1 = ap.getSeq1();
				_seq1Name = ap.getSeq1Name();
				_seq1Desc = ap.getSeq1Description();
			}

			if ( _seq2.equals("") )
			{
				_seq2 = ap.getSeq2();
				_seq2Name = ap.getSeq2Name();
				_seq2Desc = ap.getSeq2Description();
			}

			// make sure all of the params are talking about the same
			// seqs
			if ( !ap.getSeq1().equals( _seq1 ) || 
			     !ap.getSeq2().equals( _seq2 ) ) 
				throw new Exception("Seqs in new param set do not match " + 
				                    "current seqs.");

			_params.put( ap.getKey(), ap );
		}

		// create edges based on robustness
		Vector robust = _asr.getRobust();
		for ( int i = 0; i < robust.size(); i++ )
		{
			StringTokenizer t = new StringTokenizer((String)robust.get(i),"|");
			String pk = t.nextToken();
			String n = t.nextToken();
			int r = Integer.parseInt( t.nextToken() );

			AlignmentParameters ap = getAlignmentParameters( pk );
			int maxRobust = ap.getMaxRobust();
			int minRobust = ap.getMinRobust();
			r -= minRobust;
			double range = maxRobust - minRobust;
			double rob = (double)r/range;	
			
			EdgeKey ed = new EdgeKey ( n, rob, pk );
			//System.out.println("ed  " + ed.toString() );
			_allEdges.put( ed, ed );	
		}

		// add alignments (must happen after edges)
		Vector aligns = _asr.getAlignments();
		for ( int i = 0; i <  aligns.size(); i++ )
			_a.add( new Alignment((String) aligns.get(i), this) );


		} catch (Exception e) { e.printStackTrace(); }

		_currentAlignment = (Alignment)(_a.get(_curr));
	}


	public AlignmentParameters getAlignmentParameters( String key )
		throws Exception
	{

		if ( !_params.containsKey( key ) )
			throw new Exception("Couldn't find param key: " + key );

		return (AlignmentParameters)_params.get(key);
	}

	public Alignment getCurrentAlignment()
	{ return _currentAlignment; }

	// For the EdgeContainer interface
	public Vector getCurrentEdges()
	{ return _currentAlignment.getEdges(); }

	public Set getAllEdges()
	{ return _allEdges.keySet(); }

	public String getSeq1()
	{ return _seq1; }

	public String getSeq2()
	{ return _seq2; }

	public String getSeq1Name()
	{ return _seq1Name; }

	public String getSeq2Name()
	{ return _seq2Name; }

	public String getSeq1Description()
	{ return _seq1Desc; }

	public String getSeq2Description()
	{ return _seq2Desc; }

	public int getCurrentIndex() { return _curr; }

	public void addAlignment( Alignment a )
	{ _a.add( a ); }

	public Vector getAlignments()
	{ return _a; }
	
	public int getNumAlignments()
	{ return _a.size(); }

	public String getOrder()
	{ return _order; } 

	public void setOrder(String o)
	{ _order = o; } 

	public void incrementCurrent(ConditionHandler ch) 
	{ 
		int safety = _a.size();
		do
		{
			_curr++; 
			if ( _curr >= _a.size() )
				_curr = 0;

			if ( safety-- < 0 )
			{
				System.out.println("Bad condition selected!  " +
								   "Unselect it to continue." );
				_curr = 0;
				break;
			}
		}
		while ( !ch.conditionsMet( _curr ) );
		_currentAlignment = (Alignment)(_a.get(_curr));
	}

    public void decrementCurrent(ConditionHandler ch)
	{
		int safety = _a.size();
		do
		{
			_curr--;
			if (_curr <= 0)
				_curr = _a.size() - 1;

			if ( safety-- < 0 )
			{
				System.out.println("Bad condition selected!  " +
								   "Unselect it to continue." );
				_curr = 0;
				break;
			}
		}
		while ( !ch.conditionsMet( _curr ) );
		_currentAlignment = (Alignment)(_a.get(_curr));
	}

	public void setCurrent(int x)
	{
		if (x < 0)
			_curr = 0;
		else if (x >= _a.size())
			_curr = _a.size() - 1;
		else
			_curr = x;

		_currentAlignment = (Alignment)(_a.get(_curr));
	}


	public float getOptimalPercentIdentity()
	{ return _optimalPercentIdentity; }

	public float getOptimalPercentSimilarity()
	{ return _optimalPercentSimilarity; }

	public int getCountSize()
	{
		return _allEdges.size();
	}

	public int getCount( EdgeKey key )
	{
		return ((EdgeKey)_allEdges.get(key)).getCount();
	}
								 
	public double getRobustness( EdgeKey key )
	{
		return ((EdgeKey)_allEdges.get(key)).getRobustness();
	}
								 
	public double getRelativePosition( EdgeKey key )
	{
		return ((EdgeKey)_allEdges.get(key)).getCumulativePosition();
	}

	public int getMaxCount()
	{ return _maxCount; }

	public int getMaxLength() { return _maxLength; }

	public void updateMaxLength( int newLen ) 
	{ 
		_maxLength = Math.max( newLen, _maxLength ); 
	}

	public EdgeKey updateEdges( EdgeKey key, double relPos )
	{
		if ( _allEdges.containsKey( key ) )
		{
			EdgeKey e = (EdgeKey)_allEdges.get( key );
			e.increment();
			e.updateRp( relPos );
			//System.out.println("old " + e.toString() );
			_maxCount = Math.max( _maxCount, e.getCount() ); 

			return e; 
		}
		else
		{
			EdgeKey newEdge = (EdgeKey)key.clone();
			newEdge.increment();
			newEdge.updateRp( relPos );
			//System.out.println("new " + newEdge.toString() );
			_allEdges.put( newEdge, newEdge );

			return newEdge;
		}
	}
/*
	public EdgeKey updateEdges( EdgeKey key, double relPos )
	{
		if ( _counts.containsKey( key ) && _rp.containsKey( key ) )
		{
			Counter ct = (Counter)_counts.get( key );
			ct.increment();
			_maxCount = Math.max( _maxCount, ct.getCount() ); 
			((CumulativePosition)_rp.get( key )).update( relPos );

		}
		else
		{
			EdgeKey newKey = (EdgeKey)key.clone();
			_counts.put( newKey, new Counter( 1 ) );
			_rp.put( newKey, new CumulativePosition( relPos ) );
				
			_keyMap.put( newKey, newKey );

		}
	}
*/
	public void updatePercentIdentity( float percentIdentity ) 
	{
		_optimalPercentIdentity = Math.max( _optimalPercentIdentity,
					                        percentIdentity );
	}

	public void updatePercentSimilarity( float percentSimilarity ) 
	{
		_optimalPercentSimilarity = Math.max( _optimalPercentSimilarity,
					                        percentSimilarity );
	}

	public void reinitialize()
	{
	}

	public Map getParams()
	{
		return _params;
	}

	public AlignmentSetReader getAlignmentSetReader()
	{
		return _asr;
	}

}

		
