
/*
 * 
 *  file:  ./src/edu/virginia/bioch/nopt/display/AlignmentDisplayPanel.java
 * 
 *  Copyright (c) 2004,  the University of Virginia.
 *  All rights reverved.
 * 
 *  See the file COPYRIGHT in the top directory of this distribution for
 *  more information.
 *  
 *  THE SOFTWARE IS PROVIDED _AS IS_, WITHOUT WARRANTY OF ANY KIND, EXPRESS 
 *  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.  
 *  
 */ 

package edu.virginia.bioch.nopt.display;



import java.awt.*;
import java.awt.event.*;
import java.awt.font.*;
import java.awt.Font.*;
import java.awt.Graphics2D.*;
import java.awt.geom.*;
import java.awt.geom.Point2D.*;
import java.awt.geom.Rectangle2D.*;
import java.awt.geom.Line2D.*;
import javax.swing.*;
import javax.swing.event.*;
import java.text.*;
import java.util.*;

import edu.virginia.bioch.nopt.display.*;
import edu.virginia.bioch.nopt.options.*;
import edu.virginia.bioch.nopt.conditions.*;
import edu.virginia.bioch.nopt.conditions.types.*;
import edu.virginia.bioch.nopt.alignments.*;

import edu.virginia.bioch.util.*;



public class AlignmentDisplayPanel extends JPanel
	implements  ItemListener, ComponentListener, MouseListener, 
	            MouseMotionListener
{
	protected boolean displaySteady;
	protected boolean displayNames;
	protected Vector optionVec;	

	protected FontMetrics normalFontMetrics;
	protected Font normalFont;

	protected Font smallFont;
	protected String smallFontName;
	protected int smallFontSize;
	protected FontMetrics smallFontMetrics;

	protected int maxLength;
	protected TreeSet counts;
	protected String gap;
	protected int glyphWidth;
	protected double[] cutOffs;
	protected float leftOffset;
	protected float topOffset;
	protected float seq2Offset;
	protected float nameOffset;
	protected int totalRowHeight;
	protected int numCharsAcross;
	protected int separation;
	protected AffineTransform origTransform;
	protected AffineTransform position;

	protected AttributedString seq1Name;
	protected AttributedString seq2Name;

	protected Dimension currentSize;
	
	protected AlignmentHandler ah;

	protected Rectangle constrainRect;
	protected int constrainY;
	protected int constrainX;
	protected double[] currentXPos;
	protected double[] currentYPos;

	public AlignmentDisplayPanel( OptionHandler oh, 
			                      FontHandler fh, 
								  AlignmentHandler ah,
								  Dimension d )
	{
		System.out.println("AlignmentDisplayPanel constructor");

		this.ah = ah;

		gap = "-";

		displaySteady = false;
		displayNames = false;

		optionVec = oh.getOptionVector(); 
		maxLength = ah.getMaxLength();

		constrainRect = new Rectangle();

		origTransform = null;
		position = new AffineTransform();

		normalFont = fh.getNormalFont();
		smallFont = fh.getSmallFont();
		smallFontName = fh.getSmallFontName();
		smallFontSize = fh.getSmallFontSize();

		normalFontMetrics =  this.getFontMetrics( normalFont );
		smallFontMetrics = this.getFontMetrics( smallFont );

		int nameLengthLimit = 10;
		String s1 = ah.getSeq1Name();
		int s1end = Math.min( s1.length(), nameLengthLimit );
		seq1Name = new AttributedString( s1.substring(0, s1end) );
		seq1Name.addAttribute(TextAttribute.FAMILY, smallFontName);
		seq1Name.addAttribute(TextAttribute.SIZE,
				new java.lang.Float(smallFontSize));
		seq1Name.addAttribute(TextAttribute.FOREGROUND, Color.blue );

		String s2 = ah.getSeq2Name();
		int s2end = Math.min( s2.length(), nameLengthLimit );
		seq2Name = new AttributedString( s2.substring(0, s2end) );
		seq2Name.addAttribute(TextAttribute.FAMILY, smallFontName);
		seq2Name.addAttribute(TextAttribute.SIZE,
			new java.lang.Float(smallFontSize));
		seq2Name.addAttribute(TextAttribute.FOREGROUND, Color.blue );

		// 
		// Make all offsets, separations and sizes functions of 
		// the glyph size.
		//
		glyphWidth = Option.getGlyphWidth();
		separation = Option.getSeparation();
		leftOffset = (int)(2*glyphWidth);
		topOffset = (int)(2*normalFontMetrics.getMaxAscent());
		seq2Offset = normalFontMetrics.getMaxAscent() +  
					 normalFontMetrics.getMaxDescent() +
					 separation; 

		nameOffset = Option.getSmallGlyphWidth() * nameLengthLimit;
	
		totalRowHeight = 2*( normalFontMetrics.getMaxAscent() +
						     normalFontMetrics.getMaxDescent() +
						     smallFontMetrics.getMaxAscent() +
						     smallFontMetrics.getMaxDescent() ) +
						     separation;

		totalRowHeight *= 1.1;  // widen a little bit

		// set the size of the screen
		currentSize =  sizeScreen(d);
		setPreferredSize( currentSize );

		addComponentListener( this );
		addMouseListener( this );
		addMouseMotionListener( this );
	}

	protected Dimension sizeScreen( Dimension d )
	{
		//
		// Calculate the cutoff points for lines and the new dimensions.
		//
		int screenWidth = (int)(d.getWidth());

		numCharsAcross = (int)((screenWidth - 2*leftOffset )/glyphWidth);

		if ( displayNames )
			numCharsAcross -= 3;

		int numRows = (int)((double)maxLength/(double)numCharsAcross) + 1; 
		cutOffs = new double[ numRows ]; 
		for ( int i = 0; i < numRows; i++ )
			cutOffs[i] = (double)((i+1)*numCharsAcross)/(double)maxLength;

		return new Dimension( screenWidth, (int)(topOffset + 
					          (numRows * totalRowHeight)));
	}

	public void paintComponent(Graphics g)
	{
		super.paintComponent(g);
		//setBackground(new Color(240,240,255));
		setBackground(Color.white);
	
		Graphics2D g2 = (Graphics2D)g;

		Color orig = g2.getColor();
		g2.setColor(Color.lightGray);
		g2.fill( constrainRect );
		g2.setColor( orig );

		_display( g2 );
	}

	private void _display( Graphics2D g)
	{
		long start = System.currentTimeMillis();

		g.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
		                    RenderingHints.VALUE_ANTIALIAS_ON);

		Alignment sa = ah.getCurrentAlignment(); 
		String seq1 = sa.getSeq1();
		String seq2 = sa.getSeq2();

		seq1 = seq1.toLowerCase();
		seq2 = seq2.toLowerCase();

		FontRenderContext frc = g.getFontRenderContext();
	
		GlyphVector gv1 = normalFont.createGlyphVector(frc,seq1);
		GlyphVector gv2 = normalFont.createGlyphVector(frc,seq2);

		//
		// Calculate the position for each glyph.  Both vectors have the
		// same number of glyphs
		//

		Vector keys = ah.getCurrentEdges();

		currentXPos = new double[ gv1.getNumGlyphs() ];
		currentYPos = new double[ gv1.getNumGlyphs() ];

		int s1Count = 0;
		int s2Count = 0;

		int cutOffIndex = 0;
		double offset = 0.0;

		double prevXPos = -0.0001; // so the first position doesn't update

		boolean beginNewLine = true;

		for ( int i = 0; i < gv1.getNumGlyphs(); i++ )
		{
			String char1 = seq1.substring(i,i+1);
			String char2 = seq2.substring(i,i+1);

			if ( !char1.equals(gap) )
				s1Count++;
			if ( !char2.equals(gap) )
				s2Count++;

			Point2D pos1 = gv1.getGlyphPosition(i);
			Point2D pos2 = gv2.getGlyphPosition(i);

			double xPos = 0.0; 
			double yPos = 0.0;

			EdgeKey key = (EdgeKey)keys.get(i);

			// steady placement or normal
			if ( displaySteady ) 
			{
				xPos = ah.getRelativePosition( key );

				if ( xPos > cutOffs[ cutOffIndex ] )
				{
					//offset = cutOffs[ cutOffIndex ] * maxLength * glyphWidth; 
					offset = xPos * maxLength * glyphWidth; 
					cutOffIndex++; 
					beginNewLine = true;
				}
				else if ( i > 0 )
					beginNewLine = false;

			
				yPos = cutOffIndex * totalRowHeight; 
				xPos = xPos * maxLength * glyphWidth; 

				if ( xPos <= prevXPos )
				{
					System.out.print("AlignmentDisplayPanel updating xpos: " +
							           xPos );
					xPos = prevXPos + glyphWidth/2;
					System.out.println("  to: " + xPos );
				}

				prevXPos = xPos;

				xPos -= offset;

			}
			else
			{
				yPos = (((int)(i/numCharsAcross))*totalRowHeight); 
				xPos = (i%numCharsAcross)*glyphWidth; 

				if ( i%numCharsAcross == 0 )
					beginNewLine = true;
				else
					beginNewLine = false;
			}

			// draw the names if so desired
			if ( displayNames ) 
			{
				xPos += nameOffset;
				if ( beginNewLine ) 
				{
					g.drawString(seq1Name.getIterator(), 0f, 
									(float)(yPos + topOffset) );
					g.drawString(seq2Name.getIterator(), 0f, 
					            	(float)(yPos + seq2Offset + topOffset) );
				}
			}
	
			pos1.setLocation(xPos, yPos);
			pos2.setLocation(xPos, yPos + seq2Offset );

			currentXPos[i] = xPos;
			currentYPos[i] = yPos;

			gv1.setGlyphPosition(i,pos1);
			gv2.setGlyphPosition(i,pos2);

			// transform the graphics so that the options will display
			// in the proper locations
			origTransform = g.getTransform();
			position.translate( xPos+leftOffset, yPos+topOffset );
			g.transform( position );

			// draw options
			for ( int k = 0; k < optionVec.size(); k++ )
			{
				Option o = (Option)(optionVec.get(k));
				o.draw(key, i, xPos+leftOffset, yPos+topOffset, 
				   		    g, char1, char2, s1Count, s2Count );
			}						

			// set the transform back to normal
			g.setTransform( origTransform );
			position.setToIdentity();
		}

		//
		// Draw glyphs
		//
		g.drawGlyphVector(gv1,leftOffset,topOffset);
		g.drawGlyphVector(gv2,leftOffset,topOffset);

		long end = System.currentTimeMillis();
		System.out.println("AlignmentDisplayPanel    display duration: " + 
				           Long.toString(end - start));
	}


	public void itemStateChanged(ItemEvent e)
	{
		JCheckBox source = (JCheckBox)e.getItemSelectable();
		String name = source.getName();

		if ( name.equals("Steady Display") )
		{
			if ( !displaySteady )
				displaySteady = true;	
			else
				displaySteady = false;	
		}
		else if ( name.equals("Names") )
		{
			if ( !displayNames )
				displayNames = true;	
			else
				displayNames = false;

			sizeScreen( getSize() );
		}

		// don't repaint -- let SeqComparePanel deal with that.
	}


	public void componentResized(ComponentEvent e) 
	{
		Dimension d = getSize(); 
		if ( currentSize != null && !d.equals( currentSize ) )
			sizeScreen( d ); 
	}

	public void componentShown(ComponentEvent e) {};
	public void componentHidden(ComponentEvent e) {};
	public void componentMoved(ComponentEvent e) {};
	public void mouseEntered(MouseEvent e) {};
	public void mouseExited(MouseEvent e) {};
	public void mouseMoved(MouseEvent e) {};

	public void mouseDragged(MouseEvent e)
	{
		updateRect(e);
	}

	public void mousePressed(MouseEvent e)
	{
		initRect(e);
	}

	public void mouseReleased(MouseEvent e) 
	{
		if ( updateRect(e) )
			createCondition();
	}

	public void mouseClicked(MouseEvent e)
	{
		updateRect(e);
	}

	protected boolean updateRect(MouseEvent e)
	{
		if ( !e.isAltDown() )
			return false;

		int x = e.getX();
		int y = e.getY();
		
		int width = Math.abs(x - constrainX);
		int height = Math.abs(y - constrainY);

		x = Math.min(x,constrainX);
		y = Math.min(y,constrainY);

		constrainRect.setRect(x,y,width,height);
		repaint();
		//System.out.println("Update Rect: " + constrainRect.toString() );
		return true;
	}

	protected void initRect(MouseEvent e)
	{
		if ( ! e.isAltDown() )
			return;

		constrainX = e.getX();
		constrainY = e.getY();
	}

	public void createCondition()
	{
		Object[] options = { "YES", "NO" };
		int r = JOptionPane.showOptionDialog(this, 
			"Would you like to create a filter that only include " + 
			"alignments with the selected edges?", 
			"Create a filter?",
			JOptionPane.YES_NO_OPTION, 
			JOptionPane.QUESTION_MESSAGE, 
			null, 
			options, 
			options[0]);

		if ( r == 1 ) 
			return;

		// see if any edges intersect with the box we've drawn
		Vector keys = ah.getCurrentEdges();
		Vector only = new Vector();

		System.out.println( "Rect " + constrainRect.toString() ); 
		for ( int i = 0; i < currentXPos.length; i++ )
			if ( constrainRect.contains( currentXPos[i] + glyphWidth,
			                             currentYPos[i] + topOffset ) )
				only.add( (EdgeKey)keys.get(i) );

		if ( only.size() <= 0 )
		{
	       JOptionPane.showMessageDialog( this, 
			                "Selection contains no edges!",
                            "Filter Create ERROR", JOptionPane.ERROR_MESSAGE);
			return;
		}

		// ok, now create the condition and let interested parties know.
		Condition c = null;	
		try {

		EdgeKey first =  (EdgeKey)only.get(0);
		EdgeKey second =  (EdgeKey)only.get(only.size()-1);
		String desc = first.getName();
		desc += " through ";
		desc += second.getName();

		Range top = new Range( first.getEndX(), second.getEndX() );
		Range bot = new Range( first.getEndY(), second.getEndY() );

		c = new LikeThisCondition("Edges: " + desc, desc, only);

		} catch (Exception ex) {
	          JOptionPane.showMessageDialog( this, ex.getMessage(),
                            "Filter Create ERROR", JOptionPane.ERROR_MESSAGE);
			ex.printStackTrace();
		}

		if ( c != null )
			firePropertyChange("add new condition to list",null,c);

	/*
		TriangleOption to = new TriangleOption(desc,top.toSimpleString(),
		                                       Option.seq1Id(),
											   ah, desc);
		TriangleOption bo = new TriangleOption(desc,bot.toSimpleString(),
		                                       Option.seq2Id(),
											   ah, desc);
	*/
	}
}



