
/****************************************************************************** 
 * 
 *  file:  SaqiSternbergGenerator.cpp
 * 
 *  Copyright (c) 2003,  University of Virginia..
 *  All rights reverved.
 * 
 *  See the file COPYRIGHT in the top directory of this distribution for
 *  more information.
 *  
 *  THE SOFTWARE IS PROVIDED _AS IS_, WITHOUT WARRANTY OF ANY KIND, EXPRESS 
 *  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.  
 *  
 *****************************************************************************/ 


#include "SaqiSternbergGenerator.h"

namespace NOPT {

//--------------------------------------------------------------------------
//
//--------------------------------------------------------------------------
SaqiSternbergGenerator::SaqiSternbergGenerator(int gapCreate,
				                       int gapExtend,
				                       ScoringMatrix* sm,
									   bool isDNA,
									   const string& seq1,
									   const string& seq2,
									   float lowerBoundPercent,
									   float upperBoundPercent,
									   int debug,
									   int limit)
									   
: AlignmentGenerator( gapCreate, gapExtend, sm, isDNA, seq1, seq2,
					  lowerBoundPercent, upperBoundPercent, debug, false,
					  "Saqi-Sternberg" ),
  _numToGenerate(limit),
  _numAlignments(0)
{ 
	_generate();
}

//--------------------------------------------------------------------------
//
//--------------------------------------------------------------------------
SaqiSternbergGenerator::~SaqiSternbergGenerator()
{ }

//--------------------------------------------------------------------------
//
//--------------------------------------------------------------------------
void SaqiSternbergGenerator::_generate()
{

	string s1 = " " + _seq1;
	string s2 = " " + _seq2;
	int count = 0;
	int controlCount = 0;

	_scores.resize( s1.length() );
	for ( int i = 0; (unsigned int)i < _scores.size(); i++ )
		_scores[i].resize( s2.length() );

	_directions.resize( s1.length() );
	for ( int i = 0; (unsigned int)i < _directions.size(); i++ )
		_directions[i].resize( s2.length() ); 

	int duplicates = 0;
	Alignment prev;

	// controlCount allows us to keep trying alignments once the alignments
	// created start getting bad and duplicates gets us out of the loop
	// if we get into a rut.
	while ( count < _numToGenerate && 
			controlCount < 2*_numToGenerate &&
			duplicates < 10 )
	{
		_fill( s1, s2, _scores, _directions, _weights ); 

		if ( _debug & DEBUG_MORE )
		{
			_showMatrix( _scores, 0, 0, "scores" );
			_showMatrix( _directions, 0, 0, "directions" );
		}

		Alignment a = _createAlignment();

		// Set values based on the first alignment. 
		if ( count == 0 )
		{
			_optimalScore = a.getScore();
			_calculateBounds();
			prev = a;
		}

		// Only add the alignment if the score is within bounds
		// and it isn't a duplicate.
		if ( a.getScore() >= _lowerBound && 
		     a.getScore() <= _upperBound &&
			 find(_alignments.begin(), _alignments.end(), a) == 
			 _alignments.end() )
		{
			count++;
			_alignments.push_back(a);
		}
		else
		{
			if ( _debug & DEBUG_SOME )
				cout << "failed alignment: " << a << endl;

			if ( prev == a )
				duplicates++;
			else
				duplicates = 0;
		}

		// Update weights regardless of whether the alignment has been 
		// added or not.  If we don't we keep getting the same alignment
		// over and over.
		_updateWeights( a );
		controlCount++;
		prev = a;
	}

	if ( _debug & DEBUG_SOME )
		cout << "Exited loop on count: " << count 
			<< "  controlCount: " << controlCount 
			<< "  duplicates: " << duplicates << endl;
}

//--------------------------------------------------------------------------
//
//--------------------------------------------------------------------------
void SaqiSternbergGenerator::_updateWeights(const Alignment& a )
{
	string s1 = a.getSeq1();
	string s2 = a.getSeq2();

	int x = 0;
	int y = 0;

	for ( int i = 0; (unsigned int)i < s1.length(); i++ )
	{
		int sc = -(_gapCreate + _gapExtend); 
		if ( s1[i] != Alignment::gap && s2[i] != Alignment::gap )
		{
			x++;
			y++;
			sc = _sm->score(s1[i],s2[i]);
		}
		else if ( s1[i] == Alignment::gap && s2[i] != Alignment::gap )
			y++;
		else if ( s1[i] != Alignment::gap && s2[i] == Alignment::gap )
			x++;

		string key = _createKey( x, y, s1[i], s2[i] );

		sc = 3*sc/4;

		_weights[ key ] -= sc; 
	}
}

//--------------------------------------------------------------------------
//
//--------------------------------------------------------------------------
Alignment& SaqiSternbergGenerator::_createAlignment()
{
	int bi = _seq1.length() - 1;
	int bj = _seq2.length() - 1;

	string seq1; 
	string seq2; 

	while ( bi >= 0 || bj >= 0 )
	{
		int dir = _directions[bi+1][bj+1].getDirection();

		if ( dir & DIAG )
		{
			seq1 = _seq1[bi] + seq1;
			seq2 = _seq2[bj] + seq2;
			if ( _debug & DEBUG_MORE )
				cout << "DIAG " << bi+1 << " " << bj+1 
					 << " seq1 " << seq1 << "   seq2 " << seq2 << endl;
			bi--;
			bj--;
		}
		else if ( dir & DOWN )
		{
			seq1 = _seq1[bi] + seq1;
			seq2 = Alignment::gap + seq2;
			if ( _debug & DEBUG_MORE )
				cout << "DOWN " << bi+1 << " " << bj+1 
					 << " seq1 " << seq1 << "   seq2 " << seq2 << endl;
			bi--;
		}
		else if ( dir & ACROSS )
		{
			seq1 = Alignment::gap + seq1;
			seq2 = _seq2[bj] + seq2;
			if ( _debug & DEBUG_MORE )
				cout << "ACROSS " << bi+1 << " " << bj+1 
					 << " seq1 " << seq1 << "   seq2 " << seq2 << endl;
			bj--;
		}
		else if ( dir & EXTENDED_DOWN )
		{
			int beginDown = _directions[bi+1][bj+1].getDownIndex();
			int diff = bi+1 - beginDown;
			if ( _debug & DEBUG_MORE )
				cout << "diff   " << diff << " beginDown " << beginDown <<
					 "  bi+1 " << bi + 1 << 
					 "  dir obj " << _directions[bi+1][bj+1] << endl;

			for ( int x = 0; x < diff; x++ )
			{
				seq1 = _seq1[bi] + seq1;
				seq2 = Alignment::gap + seq2;
				if ( _debug & DEBUG_MORE )
					cout << "EXTENDED_DOWN " << bi+1 << " " << bj+1 
						 << " seq1 " << seq1 << "   seq2 " << seq2 << endl;
				bi--;
			}
		}
		else if ( dir & EXTENDED_ACROSS )
		{
			int beginAcross = _directions[bi+1][bj+1].getAcrossIndex();
			int diff = bj+1 - beginAcross;
			if ( _debug & DEBUG_MORE )
				cout << "diff   " << diff << " beginAcross " << beginAcross <<
					 "  bj+1 " << bj + 1 << 
					 "  dir obj " << _directions[bi+1][bj+1] << endl;

			for ( int x = 0; x < diff; x++ )
			{
				seq1 = Alignment::gap + seq1;
				seq2 = _seq2[bj] + seq2;
				if ( _debug & DEBUG_MORE )
					cout << "EXTENDED_ACROSS " << bi+1 << " " << bj+1 
						 << " seq1 " << seq1 << "   seq2 " << seq2 << endl;
				bj--;
			}
		}
		else
			throw(GenericException((string)"Bad direction in forward " +
				  (string)"direction matrix at bi:" + tos(bi) + " bj:" + 
                  tos(bj) + "  dir: " + tos(dir) ));

	}

	// Now actually create the alignment
	//
	// Note that because we're using the "adjusted" score matrix, the score
	// it returns isn't the real alignment score.  Therefore Alignment must
	// calculate the score based on the gapped sequences.
	Alignment* a = new Alignment( seq1, seq2 , *_sm, _gapCreate, _gapExtend ); 

	return *a;
} 

}
