

/****************************************************************************** 
 * 
 *  file:  SmootGenerator.h
 * 
 *  Copyright (c) 2003,  University of Virginia..
 *  All rights reverved.
 * 
 *  See the file COPYRIGHT in the top directory of this distribution for
 *  more information.
 *  
 *  THE SOFTWARE IS PROVIDED _AS IS_, WITHOUT WARRANTY OF ANY KIND, EXPRESS 
 *  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.  
 *  
 *****************************************************************************/ 


#ifndef SMOOTGENERATOR_H
#define SMOOTGENERATOR_H

#include <string>
#include <cstdlib>
#include <algorithm>
#include <fstream>
#include <vector>
#include <list>
#include <ScoringMatrix.h> 
#include <GenericException.h> 
#include <XMLable.h> 
#include <XML.h> 
#include <Tostring.h> 
#include <time.h> 
#include "Alignment.h" 
#include "AlignmentGenerator.h" 
#include "MyersMillerGenerator.h" 

using namespace std;

namespace NOPT {

/**
 * Generates a set of near optimal alignments according to the Smoot algorithm.
 * This algorithm is a variation builds on the Zuker algorithm, but does
 * so in linear space.
 */
class SmootGenerator : public AlignmentGenerator
{
	public:
				
		/**
		 * Constructor.
		 * Constructs the score and direction matrices and pushes the
		 * alignments created onto the _alignments vector.
		 * \param int gapCreate - Gap create penalty.
		 * \param int gapExtend - Gap extend penalty.
		 * \param ScoringMatrix* sm - Pointer to a ScoringMatrix object.
		 * \param bool isDNA - Whether the sequences are DNA.
		 * \param string seq1 - First sequence to be aligned.
		 * \param string seq2 - Second sequence to be aligned.
		 * \param float lowerbound - Lower bound percentage, between 0 and 1.
		 * \param float upperbound - Upper bound percentage, between 0 and 1.
		 * \param int debug - debug level. 
		 */
		SmootGenerator(int gapCreate, int gapExtend, ScoringMatrix* sm,
						   bool isDNA, const string& seq1, 
						   const string& seq2, float lowerbound, 
						   float upperbound, int debug);

		void getAscii(bool compress = true);

	private:

		/**
		 * Generates the sample of alignments. 
		 */
		void _generate(bool uncompress);
		
		/**
		 * Returns the optimal score for the two seqs and does so in
		 * linear space.
		 */
		int _calcOptimalScore(const string& a, const string& b);

		/**
		 * Creates a Myers-Miller alignment of seqs a and b and puts the 
		 * respective aligned seqs in resultA and resultB.
		 */
		int _myersMillerAlign(const string& a, const string& b,
		                       string& resultA, string& resultB);

		vector<int> _sums;

		void _rememberEdges( const string& a, const string& b, int score );

		map<string,int> usedEdges;

};

}

#endif
