
/****************************************************************************** 
 * 
 *  file:  noptalign.cpp
 * 
 *  Copyright (c) 2003,  University of Virginia..
 *  All rights reverved.
 * 
 *  See the file COPYRIGHT in the top directory of this distribution for
 *  more information.
 *  
 *  THE SOFTWARE IS PROVIDED _AS IS_, WITHOUT WARRANTY OF ANY KIND, EXPRESS 
 *  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.  
 *  
 *****************************************************************************/ 


#include "MyersMillerGenerator.h"
#include "ZukerGenerator.h"
#include "WatermanByersGenerator.h"
#include "SaqiSternbergGenerator.h"
#include <ScoringMatrix.h>
#include <SequenceReader.h>
#include <tclap/CmdLine.h>
#include <iostream>

using namespace NOPT;
using namespace TCLAP;
using namespace std;

//namespace NOPTALIGN {


int _gapCreate;
int _gapExtend;
bool _isDna;
string _smName;
string _seq1Filename;
string _seq2Filename; 
float _upperBound; 
float _lowerBound; 
int _limit;
int _numToGenerate;
int _debug;
string _order;
bool _xml;
bool _local;
bool _all;
bool _uncompress;
bool _ss;
bool _mm;
bool _cc;
bool _CC;
bool _Rx;
bool _r;

void parseOptions(int argc, char** argv);

int main(int argc, char** argv)
{
	try {

	parseOptions(argc,argv);

	transform( _smName.begin(), _smName.end(), _smName.begin(), ::tolower );
	if ( _isDna && _smName != "dna" )
		_smName = "dna";
	
	ScoringMatrix* pam = new ScoringMatrix(_smName);

	string seq1; 
	string header1; 
	SequenceReader::readFasta(_seq1Filename,seq1,header1);

	string seq2;
	string header2;
	SequenceReader::readFasta(_seq2Filename,seq2,header2);

	clock_t t1 = clock();

	AlignmentGenerator *ag;

	// choose the generator to use
	if ( _ss )
		ag = new SaqiSternbergGenerator(_gapCreate,_gapExtend,pam,
				                        _isDna, seq1, seq2,
					                    _lowerBound,_upperBound, _debug, 
								        _numToGenerate); 
	else if ( _mm )
		ag = new MyersMillerGenerator(_gapCreate,_gapExtend,pam,
				                      _isDna, seq1, seq2,
					                  _debug, _local); 
	else if ( _all )
		ag = new WatermanByersGenerator (_gapCreate,_gapExtend,pam,_isDna, 
				                         seq1, seq2, _lowerBound,_upperBound, 
								         _debug, _local); 
	else
		ag = new ZukerGenerator(_gapCreate,_gapExtend,pam,_isDna, seq1, seq2,
					            _lowerBound,_upperBound, _limit,_debug,_order,
						        _local); 

	// now generate the appropriate output
	if ( _cc )
		ag->getCounts( false );

	else if ( _CC )
		ag->getCounts( true );

	else if ( ag->xmlable() && _xml )
		cout << ag->getXml() << endl;

	else
		ag->getAscii(_uncompress);

	// check if we should get robustness
	if ( _Rx || _r )
	{
		ZukerGenerator* zg;

		// We need Zuker if we want robustness.
		if ( _ss || _mm || _all )
			zg = new ZukerGenerator(_gapCreate,_gapExtend,pam,_isDna, seq1, 
					                seq2, _lowerBound,_upperBound, _limit,
									_debug,_order, _local); 
		else
			zg = (ZukerGenerator*)ag;

		if ( _Rx )
			zg->getRobustness(false);
		else 
			zg->getRobustness(true);
	}

	clock_t t2 = clock();

	if ( _debug & DEBUG_TIME )
		cout << "Time elapsed: " << t2 - t1 << endl;


	} catch (GenericException e) { cout << "ERROR " << e.errorData() << endl; }

	return 0;
}


void parseOptions(int argc, char** argv)
{
	//
	// Define all possible command line arguments.  Note that last arg
	// in the constructor is the default value for that particular 
	// argument.
	//
	string message = 
"    This tool is used to generate near optimal alignments \n    of two biological sequences.  The default is to generate a \n    sample of alignments using the Zuker algorithm (ref), \n    although there is an option (-A) to generate ALL near optimal \n    alignments using the Waterman-Byers algorithm (ref).  \n    Users must be careful with this option as for \n    even relatively short sequences (~50 residues) and small \n    neighborhoods this option can create very large numbers of\n    alignments.";

	try { 

	CmdLine cmd(argv[0],message, "rel-1-0");
	
	ValueArg<int> gapCreate("f","gapCreate", "The cost of creating a gap", 
								 false, -10, "negative int");
	cmd.add( gapCreate );

	ValueArg<int> gapExtend("g","gapExtend",
					        "The cost for each extension of a gap", 
	                        false, -2, "negative int"); 
	cmd.add( gapExtend );

	SwitchArg dna("n","isDna","The input sequences are DNA", false); 
	cmd.add( dna );

	ValueArg<string> scoringMatrixName("s","scoringMatrix", 
					                   "Scoring Matrix name", 
							           false,"BLOSUM50","name string"); 
	cmd.add( scoringMatrixName );

	ValueArg<string> seq1Filename ("1","filename1",
					               "Sequence 1 filename (FASTA format)", 
								      true,"","filename"); 
	cmd.add( seq1Filename );

	ValueArg<string> seq2Filename ("2","filename2",
					               "Sequence 2 filename (FASTA format)", 
								      true,"","filename"); 
	cmd.add( seq2Filename );

	ValueArg<float> lowerBound("b","lowerBound", "lower percentage bound", 
							 false,1.0,"float lte 1"); 
	cmd.add( lowerBound );

	ValueArg<float> upperBound("u","upperBound", "upper percentage bound", 
							 false,1.0,"float lte 1"); 
	cmd.add( upperBound );

	ValueArg<int> limit("l","limit","Max number of alignments allowed", 
	                         false, 1000,"int"); 
	cmd.add( limit );

	ValueArg<int> numToGenerate("N","number",
					         "Number of alignments to attempt to generate", 
	                         false, 100,"int"); 
	cmd.add( numToGenerate );

	ValueArg<int> debug("d","debugLevel",
					"show debug information (bits 1,2,4,8 => levels 0-15)",
	               false, 0, "int" ); 
	cmd.add( debug );

	ValueArg<string> order("o","order","order the alignments are output in",
	               false, "CREATE","CREATE|SORT|REVERSE|RANDOM"); 
	cmd.add( order );

	SwitchArg xml("x","xml","output information as xml", false);
	cmd.add( xml );

	SwitchArg uncompress("U","uncompress","uncompress alignment string",false); 
	cmd.add( uncompress );

	SwitchArg local("L","local","local alignments", false); 
	cmd.add( local );

	SwitchArg all("A","all", "generate all near optimal alignments", false); 
	cmd.add( all );

	SwitchArg ss("S","SS", "generate Saqi-Sternberg sample", false); 
	cmd.add( ss );

	SwitchArg mm("M","MM", "creates ONE (1) Myers-Miller optimal alignment.", 
				 false); 
	cmd.add( mm );

	SwitchArg CC("C","verbose_count_edges", 
			     "counts individual edges for all alignments (with status)", 
				 false); 
	cmd.add( CC );

	SwitchArg cc("c","count_edges", 
			     "counts individual edges for all alignments (without status)",
				 false); 
	cmd.add( cc );

	SwitchArg Rx("R","all_robustness","dump the robustness for EVERY edge",
			    false);
	cmd.add(Rx);

	SwitchArg r("r","nopt_robustness",
		"dump the robustness only for edges within near optimal neighborhood",
		false);
	cmd.add(r);

	//
	// Parse the command line.
	//
	cmd.parse(argc,argv);

	//
	// Set variables
	//
	_gapCreate = gapCreate.getValue();
	_gapExtend = gapExtend.getValue();
	_isDna = dna.getValue();
	_smName = scoringMatrixName.getValue();
	_seq1Filename = seq1Filename.getValue();
	_seq2Filename = seq2Filename.getValue();
	_upperBound = upperBound.getValue();
	_lowerBound = lowerBound.getValue();
	_limit = limit.getValue();
	_numToGenerate = numToGenerate.getValue();
	_debug = debug.getValue();
	_order = order.getValue();
	_xml = xml.getValue();
	_uncompress = uncompress.getValue();
	_local = local.getValue();
	_all = all.getValue();
	_ss = ss.getValue();
	_mm = mm.getValue();
	_CC = CC.getValue();
	_cc = cc.getValue();
	_Rx = Rx.getValue();
	_r = r.getValue();


	} catch (ArgException e)  
    { 
		cerr << "error: " << e.error() << " for arg " << e.argId() << endl; 
		exit(1);
	}

}

