
/****************************************************************************** 
 * 
 *  file:  scorealign.cpp
 * 
 *  Copyright (c) 2003,  University of Virginia..
 *  All rights reverved.
 * 
 *  See the file COPYRIGHT in the top directory of this distribution for
 *  more information.
 *  
 *  THE SOFTWARE IS PROVIDED _AS IS_, WITHOUT WARRANTY OF ANY KIND, EXPRESS 
 *  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.  
 *  
 *****************************************************************************/ 


#include "Alignment.h"
#include <ScoringMatrix.h>
#include <SequenceReader.h>
#include <iostream>
#include <fstream>
#include <tclap/CmdLine.h>

using namespace NOPT;
using namespace TCLAP;
using namespace std;


int _gapCreate;
int _gapExtend;
string _smName;
string _inputFile;
bool _pir;

void parseOptions(int argc, char** argv);

int main(int argc, char** argv)
{
	try {

	parseOptions(argc,argv);

	ScoringMatrix pam(_smName);

	vector<string> v; 
	vector<string> h;

	if( _pir )
		SequenceReader::readMultiplePir(_inputFile,v,h);
	else
		SequenceReader::readMultipleFasta(_inputFile,v,h);

	cout << "seq1     " << v[0] << endl << endl;
	cout << "seq2     " << v[1] << endl << endl;

	int sc = Alignment::SWScore(v[0], v[1], pam, _gapCreate, _gapExtend, true);
	cout << "score:  " << sc << endl;

	} catch (GenericException e) { cout << "ERROR " << e.errorData() << endl; }

	return 0;
}


void parseOptions(int argc, char** argv)
{
	try { 

	//
	// Define all possible command line arguments.  Note that last arg
	// in the constructor is the default value for that particular 
	// argument.
	//
	ValueArg<int> gapCreate("f","gapCreate", "The cost of creating a gap", 
								 false, -10,"negative int");

	ValueArg<int> gapExtend("g","gapExtend",
					        "The cost for each extension of a gap", 
	                         false, -2,"negative int"); 

	ValueArg<string> scoringMatrixName("s","scoringMatrix", 
					                   "Scoring Matrix name", 
							           false,"BLOSUM50","name string"); 

	ValueArg<string> filename ("1","filename","input filename", 
								      true,"","filename"); 

	SwitchArg pir ("p","pir","files in PIR format", false);

	string message = 
"    This is a simple program that calculates the score\n    of an alignment given the alignment and scoring parameters.  It\n    is used primarily as an independent check that alignment scores\n    are calculated correctly by near optimal generation algorithms.";

	// Parse the command line.
	//
	CmdLine cmd(argv[0], message, "rel-1-0"); 
	cmd.add(gapCreate);
	cmd.add(gapExtend);
	cmd.add(scoringMatrixName);
	cmd.add(filename);
	cmd.add(pir);
	cmd.parse(argc,argv);

	//
	// Set variables
	//
	_gapCreate = gapCreate.getValue();
	_gapExtend = gapExtend.getValue();
	_smName = scoringMatrixName.getValue();
	_inputFile = filename.getValue();
	_pir = pir.getValue();

	} catch (ArgException e)  
	{ 
		cerr << "error: " << e.error() << " for arg " << e.argId() << endl; 
		exit(1);
	}
}

