
/*
 * 
 *  file:  ./src/edu/virginia/bioch/nopt/display/path/PathDisplay.java
 * 
 *  Copyright (c) 2004,  the University of Virginia.
 *  All rights reverved.
 * 
 *  See the file COPYRIGHT in the top directory of this distribution for
 *  more information.
 *  
 *  THE SOFTWARE IS PROVIDED _AS IS_, WITHOUT WARRANTY OF ANY KIND, EXPRESS 
 *  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.  
 *  
 */ 


package edu.virginia.bioch.nopt.display.path;


import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import java.lang.Thread.*;
import java.util.*;
import java.lang.Exception.*;
import java.io.IOException;
import edu.virginia.bioch.nopt.display.*;
import edu.virginia.bioch.nopt.alignments.*;
import edu.virginia.bioch.nopt.options.*;

import edu.virginia.bioch.util.*;


import java.awt.font.*;
import java.awt.Font.*;
import java.awt.geom.*;
import java.awt.Graphics2D.*;
import java.text.*;

import edu.umd.cs.piccolo.*;
import edu.umd.cs.piccolo.event.*;
import edu.umd.cs.piccolo.nodes.*;

public class PathDisplay 
	extends JPanel
	implements ActionListener
{

	protected PCanvas canvas;
	protected PFastPath allEdges;
	protected PFastPath someEdges;
	public static final int bigLength = 10000;
	public static final int offset = 20;
	public static final int charWidth = 12;
	public static final int space = 4;
	public static final int halfSpace = 2;
	public static final int edgeLen = charWidth + space;
	protected EdgeContainer ec;
	protected String seq1;
	protected String seq2;
	protected Color boxColor;

	public PathDisplay( String seq1, String seq2, EdgeContainer ec )
	{
		long start = System.currentTimeMillis();

		this.ec = ec;
		this.seq1 = seq1;
		this.seq2 = seq2;

		boxColor = new Color( 255,255,140 ); // light yellow

		setPreferredSize( new Dimension( 800, 800 ) );
		setLayout(new BorderLayout());

    	canvas = new PCanvas();
	    add(canvas,BorderLayout.CENTER);
		Font smallFont = new Font ("Times", Font.PLAIN, 8);
		Font normalFont = new Font ("Times", Font.PLAIN, 12);

		int center = edgeLen/4;

		// draw all edges
		allEdges = new PFastPath();
		Iterator it = (ec.getAllEdges()).iterator();
		while ( it.hasNext() )
		{
			EdgeKey key = (EdgeKey)it.next();
				
			int prevX = key.getBeginX();
			int prevY = key.getBeginY();
				
			int x = key.getEndX();
			int y = key.getEndY();

			allEdges.fastMoveTo( offset+(prevX*edgeLen),offset+(prevY*edgeLen));
			allEdges.fastLineTo( offset+(x*edgeLen),offset+(y*edgeLen) );
		}
		allEdges.fastFinish();
		canvas.getLayer().addChild( allEdges );
		allEdges.setStroke( new BasicStroke(1) );
		allEdges.setStrokePaint( Color.red );

		// draw some edges
		drawSomeEdges( );

		// add the top/horizontal sequence
		PCamera topCam = new PCamera();
		topCam.setBounds(0,0,800,50);
		PLayer topLay = new PLayer();
		PHorizontalSequence phor = new PHorizontalSequence( seq1, offset, 
				                   edgeLen, center, charWidth, normalFont );
		topLay.addChild( phor );
		topCam.addLayer( topLay );
		canvas.getCamera().addChild( topCam );

		// add the vertical sequence
		PCamera vertCam = new PCamera();
		vertCam.setBounds(0,0,50,800);
		PLayer vertLay = new PLayer();
		PVerticalSequence pvert = new PVerticalSequence( seq2, offset, 
				                   edgeLen, center, charWidth, normalFont );
		vertLay.addChild( pvert );
		vertCam.addLayer( vertLay );
		canvas.getCamera().addChild( vertCam );

		// clear the event handlers so zoom/pan events aren't handled twice
		canvas.removeInputEventListener( canvas.getZoomEventHandler() );
		canvas.removeInputEventListener( canvas.getPanEventHandler() );

		// add new zoom/pan event handlers
		PanHandler vp = new PanHandler( topCam, vertCam );
		canvas.addInputEventListener( vp );

		ZoomHandler vz = new ZoomHandler( topCam, vertCam );
		canvas.addInputEventListener( vz );

		// add an event handler to draw the highlight rect
		canvas.addInputEventListener( 
			new PBasicInputEventHandler() 
			{
				PPath vertRect;
				PPath horRect;

				public void mouseMoved(PInputEvent e)
				{
					// don't draw highlights if the scale is too small
					PCamera c = e.getCamera();
					if ( c.getViewScale() < 0.4 ) // "empirically" deirved...
					{
						if ( horRect != null )
						{
							canvas.getLayer().removeChild( horRect );
							horRect = null;
						}
								
						if ( vertRect != null )
						{
							canvas.getLayer().removeChild( vertRect );
							vertRect = null;
						}

						return;
					}

					double spacePercent = (double)space/(double)edgeLen;	
					Point2D point = e.getPosition();
					int xMousePos = (int)(point.getX()); 
				    int yMousePos = (int)(point.getY());

					// if the mouse point intersects allEdges do the math...
					if ( allEdges.intersects( new Rectangle2D.Double(
								             (double)xMousePos-edgeLen/2,
			                                  (double)yMousePos-edgeLen/2,
					                                    (double)edgeLen, 
														(double)edgeLen) ) ) 
					{
						float xDown = ((float)xMousePos)/((float)edgeLen);
						int xFloor = (int)(Math.floor( xDown ));
						float xRem = xDown - xFloor;

						boolean xThick;
						boolean yThick;

						if ( xRem > spacePercent )
							xThick = true;
						else
							xThick = false;

						int xClickPos = xFloor * edgeLen;

						float yDown = ((float)yMousePos)/((float)edgeLen);
						int yFloor = (int)(Math.floor( yDown ));
						float yRem = yDown - yFloor;

						if ( yRem > spacePercent )
							yThick = true;
						else
							yThick = false;

						int yClickPos = yFloor * edgeLen;
								
						if ( horRect != null )
							canvas.getLayer().removeChild( horRect );
								
						// draw the horizontal highlight rectangle
						if ( yThick && xThick )
							horRect = PPath.createRectangle( 
									    -bigLength, 
									    yClickPos+space, 
										bigLength+xClickPos+edgeLen+space, 
										edgeLen );

						else if ( !yThick && xThick )
							horRect = PPath.createRectangle( 
									    -bigLength, 
										yClickPos+halfSpace, 
										bigLength+xClickPos+edgeLen+space, 
										space );

						else if ( yThick && !xThick )
							horRect = PPath.createRectangle( 
									    -bigLength, 
										yClickPos+space, 
										bigLength+xClickPos+space, 
										edgeLen );

						else
							horRect = PPath.createRectangle( 
									    -bigLength, 
									    yClickPos+halfSpace, 
										bigLength+xClickPos+space, 
										space );

						horRect.setPaint( boxColor );
						horRect.setStrokePaint( boxColor );
						canvas.getLayer().addChild( horRect );
						horRect.moveToBack();

						// draw the vertical highlight rectangle
						if ( vertRect != null )
							canvas.getLayer().removeChild( vertRect );

						if ( xThick && yThick )
							vertRect = PPath.createRectangle( 
										xClickPos+space, 
										-bigLength, 
										edgeLen,
										bigLength+yClickPos+edgeLen+space );

						else if ( !xThick && yThick )
							vertRect = PPath.createRectangle( 
										xClickPos+halfSpace, 
										-bigLength, 
										space, 
										bigLength+yClickPos+edgeLen+space );

						else if ( xThick && !yThick )
							vertRect = PPath.createRectangle( 
										xClickPos+space, 
										-bigLength, 
										edgeLen, 
										bigLength+yClickPos+space );

						else
							vertRect = PPath.createRectangle( 
									    xClickPos+halfSpace, 
										-bigLength, 
										space, 
										bigLength+yClickPos+space );

						vertRect.setPaint( boxColor );
						vertRect.setStrokePaint( boxColor );
						canvas.getLayer().addChild( vertRect );
						vertRect.moveToBack();
					}
				}

			} );

	}

	protected void drawSomeEdges( )
	{
		if ( someEdges != null )
		{
			canvas.getLayer().removeChild( someEdges );
			someEdges.reset();
		}
		else
			someEdges = new PFastPath();

		// draw 
		Iterator it2 = (ec.getCurrentEdges()).iterator();
		while ( it2.hasNext() )
		{
			EdgeKey key = (EdgeKey)it2.next();
				
			int prevX = key.getBeginX();
			int prevY = key.getBeginY();
				
			int x = key.getEndX();
			int y = key.getEndY();

			someEdges.fastMoveTo(offset+(prevX*edgeLen),offset+(prevY*edgeLen));
			someEdges.fastLineTo(offset+(x*edgeLen),offset+(y*edgeLen));
		}
		someEdges.fastFinish();
		canvas.getLayer().addChild( someEdges );
		someEdges.setStroke( new BasicStroke(1) );
		someEdges.setStrokePaint( Color.blue );
	}

    public void actionPerformed(ActionEvent e)
    {
		// null is the timer event
		String command = e.getActionCommand();
		if ( command == null ||
			 command.equals("Step Forward") ||
			 command.equals("Step Backward") ||
			 command.equals("Location") )
			drawSomeEdges();
	}
}


