
/****************************************************************************** 
 * 
 *  file:  MyersMillerGenerator.cpp
 * 
 *  Copyright (c) 2003,  University of Virginia..
 *  All rights reverved.
 * 
 *  See the file COPYRIGHT in the top directory of this distribution for
 *  more information.
 *  
 *  THE SOFTWARE IS PROVIDED _AS IS_, WITHOUT WARRANTY OF ANY KIND, EXPRESS 
 *  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.  
 *  
 *****************************************************************************/ 

#include "MyersMillerGenerator.h"

namespace NOPT { 

//--------------------------------------------------------------------------
//
//--------------------------------------------------------------------------

MyersMillerGenerator::MyersMillerGenerator(int gapCreate,
				                          int gapExtend,
										ScoringMatrix* pam,
										bool isDNA,
										const string& seq1,
										const string& seq2,
										int debug,
										bool local)
: AlignmentGenerator( gapCreate, gapExtend, pam, isDNA, seq1, seq2,
                      1, 1, debug, local, "Myers-Miller" ),
  _aInd1(0),
  _aInd2(0),
  _alignedSeq1(""),
  _alignedSeq2("")
{
	// because I've been too lazy to fix the indices...
	//_seq1 = " " + _seq1;
	//_seq2 = " " + _seq2;

	CC.resize(_seq2.length()+1);
	DD.resize(_seq2.length()+1);
	RR.resize(_seq2.length()+1);
	SS.resize(_seq2.length()+1);

	_generate();
}

//--------------------------------------------------------------------------
//
//--------------------------------------------------------------------------
MyersMillerGenerator::~MyersMillerGenerator()
{ }


//--------------------------------------------------------------------------
//
//--------------------------------------------------------------------------
int MyersMillerGenerator::_align (const string& A, const string& B, 
                  				  int tb, int te)
{

	int optMidAcross;  // optimal midpoint across (j index)
	int optMidScore;   // score for optimal midpoint (midDown,optMidAcross) 

	int i, j, c, e, d, s, t; 

	int M = A.length();
	int N = B.length();

	//
	// Boundary cases.
	//
	if (N <= 0)
	{
		if (M > 0)
			del(M);
		return gap(M);
	}

	if (M <= 1 )
	{
		if (M <=0 )
		{
			ins(N);
			return gap(N);
		}

		if (tb < te)
			tb = te;
	  	optMidScore = (tb + _gapExtend) + gap(N);
	  	optMidAcross = 0;
	  	for (j = 1; j <= N; j++)
		{
			c = gap(j - 1) + _sm->score(A[0],B[j-1]) + gap(N - j);
		  	if (c > optMidScore)
			{
				optMidScore = c;
				optMidAcross = j;
			}
 			// arbitrary decision here... 
		}

	  	if (optMidAcross == 0)
		{
			ins(N);
			del(1);
		}
	  	else
		{
			if (optMidAcross > 1)
				ins(optMidAcross - 1);
		  	rep();
			if (optMidAcross < N)
				ins(N - optMidAcross);
		}
	  	return optMidScore;
	}

	//
	// Find optimum midpoint (midDown,optMidAcross) with score: optMidScore 
	//
	// Forward phase: 
	// Compute C(M/2,k) & D(M/2,k) for all k 
	//
	int midDown = M / 2; // midpoint down. Note that this is chosen arbitrarily
	                     // and hence isn't optimal by itself.  
						 // It doesn't matter where we divide the seqs.
	CC[0] = 0;					
	t = _gapCreate;
	for (j = 1; j <= N; j++)
	{
		CC[j] = t = t + _gapExtend;
		DD[j] = t + _gapCreate;
	}
	t = tb;
	for (i = 1; i <= midDown; i++)
	{
		s = CC[0];
		CC[0] = c = t = t + _gapExtend;
	  	e = t + _gapCreate;
	  	for (j = 1; j <= N; j++)
		{
			e = max( c + _gapExtend + _gapCreate, e + _gapExtend );
			d = max(CC[j] + _gapExtend + _gapCreate, DD[j] + _gapExtend );
		  	c = max(s + _sm->score(A[i-1],B[j-1]), max(d,e));
		  	s = CC[j];
		  	CC[j] = c;
		  	DD[j] = d;
		}
	}
	DD[0] = CC[0];

	// 
	// Reverse phase:
	// Compute R(M/2,k) & S(M/2,k) for all k 
	// 
	RR[N] = 0;					
	t = _gapCreate;						
	for (j = N - 1; j >= 0; j--)
	{
		RR[j] = t = t + _gapExtend;
		SS[j] = t + _gapCreate;
	}
  	t = te;
  	for (i = M - 1; i >= midDown; i--)
	{
	  	s = RR[N];
	  	RR[N] = c = t = t + _gapExtend;
	  	e = t + _gapCreate;
	  	for (j = N - 1; j >= 0; j--)
		{
			e = max( c + _gapExtend + _gapCreate, e + _gapExtend );
			d = max( RR[j] + _gapExtend + _gapCreate, SS[j] + _gapExtend);
		  	c = max(s + _sm->score(A[i],B[j]),max(d,e));
		  	s = RR[j];
		  	RR[j] = c;
		  	SS[j] = d;
		}
	}
  	SS[N] = RR[N];

	//  
	// Find optimal midpoint 
	//
	// Given our arbitrary midDown that we've used to calculate CC,DD and RR,SS,
	// now find the best combination of CC+RR or DD+SS.  The combination of 
	// midDown,optMidAcross is the optimal midpoint.  optMidScore is thus 
	// the optimal alignment score.
	// 
	optMidScore = CC[0] + RR[0];			
	optMidAcross = 0;
	int type = 1;
	for (j = 0; j <= N; j++) 
	{
		if ((c = CC[j] + RR[j]) >= optMidScore)
	  		if (c > optMidScore || CC[j] != DD[j] && RR[j] == SS[j] )
			{
		  		optMidScore = c;
		  		optMidAcross = j;
			}
			// arbitrary decision point here...
	}

	for (j = N; j >= 0; j--)
		if ((c = DD[j] + SS[j] - _gapCreate) > optMidScore) 
	  	{
			optMidScore = c;
			optMidAcross = j;
			type = 2;
	  	}
		// arbitrary decision point here...

	//
	// Conquer: recursively around midpoint 
	//
	if (type == 1)
	{
		_align( A.substr(0,midDown), B.substr(0,optMidAcross), tb, _gapCreate);
		_align( A.substr(midDown), B.substr(optMidAcross), _gapCreate, te);
	}
  	else
	{

		_align( A.substr(0,midDown-1), B.substr(0,optMidAcross), tb, 0);
	  	del(2);
		_align( A.substr(midDown+1), B.substr(optMidAcross), 0, te);
	}

  	return optMidScore;
}


//--------------------------------------------------------------------------
//
//--------------------------------------------------------------------------
void MyersMillerGenerator::_generate() 
{
	_createParamString();

	_optimalScore = _align(_seq1, _seq2, _gapCreate, _gapCreate);	

	Alignment a( _alignedSeq1, _alignedSeq2, _optimalScore,
				 0, 0, _seq1.length(), _seq2.length() );
	_alignments.push_back( a );

	// Sanity Check. 
	// Make sure the alignment generated adds up to the optimal score.
	int testOpt = Alignment::SWScore( a.getSeq1(), a.getSeq2(), 
					                  *_sm, _gapCreate, _gapExtend );
	if ( _optimalScore != testOpt )
		throw( GenericException((string)"Myers-Miller optimal score does not "
								"match that of the generated alignment. " +
								" Myers-Miller: " + tos(_optimalScore) + 
								"  vs. generated: " + tos(testOpt)) );
							                  
}


//--------------------------------------------------------------------------
//
//--------------------------------------------------------------------------
void MyersMillerGenerator::getAscii(bool uncompress) 
{
    cout << "p~" << _paramSum << _paramString << endl; 

    for ( int i = 0; (unsigned int)i < _alignments.size(); i++ ) // silly
    {
		cout << "a" << "~" << _paramSum << "|" 
		     << _alignments[i].getAscii(uncompress) << endl;
	}
}

//--------------------------------------------------------------------------
//
//--------------------------------------------------------------------------
Alignment MyersMillerGenerator::getAlignment()
{
	if ( _alignments.size() != 1 )
		throw( GenericException((string)"No alignments found!"));
		
	return _alignments[0];
}

//--------------------------------------------------------------------------
//
//--------------------------------------------------------------------------
void MyersMillerGenerator::del(int k)
{ 
	if ( _debug & DEBUG_LOTS )
		cout << "del " << k << endl; 

	while ( k-- > 0 )
	{
		_alignedSeq1 += _seq1[_aInd1++];
		_alignedSeq2 += "-"; 
	}

}                                                                               

//--------------------------------------------------------------------------
//
//--------------------------------------------------------------------------
void MyersMillerGenerator::ins(int k)
{
	if ( _debug & DEBUG_LOTS )
		cout << "ins " << k << endl; 

	while ( k-- > 0 )
	{
		_alignedSeq1 += "-"; 
		_alignedSeq2 += _seq2[_aInd2++];
	}
} 

//--------------------------------------------------------------------------
//
//--------------------------------------------------------------------------
void MyersMillerGenerator::rep()
{
	if ( _debug & DEBUG_LOTS )
		cout << "rep " << endl;

	_alignedSeq1 += _seq1[_aInd1++];
	_alignedSeq2 += _seq2[_aInd2++];
}

//--------------------------------------------------------------------------
//
//--------------------------------------------------------------------------
int MyersMillerGenerator::gap(int k)
{
	if ( k <= 0 )
		return 0;
	else
		return (_gapCreate + (_gapExtend * k));
}

}
