
/*
 * 
 *  file:  ./src/edu/virginia/bioch/nopt/alignments/EdgeKey.java
 * 
 *  Copyright (c) 2004,  the University of Virginia.
 *  All rights reverved.
 * 
 *  See the file COPYRIGHT in the top directory of this distribution for
 *  more information.
 *  
 *  THE SOFTWARE IS PROVIDED _AS IS_, WITHOUT WARRANTY OF ANY KIND, EXPRESS 
 *  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.  
 *  
 */ 

package edu.virginia.bioch.nopt.alignments;



import java.lang.*;
import java.util.*;


public class EdgeKey
{
	protected int beginX; 
	protected int beginY;
	protected int endX;
	protected int endY; 
	protected char topChar; 
	protected char botChar;

    protected double cumRp;
    protected double robustness;
	protected int count;
	protected String name;
	protected String paramKey;

	private int hashCode;

	public EdgeKey()
	{
		init();
	}

	public EdgeKey ( String s, double rob, String params )
	{
		parseString(s);
		robustness = rob;
		paramKey = params;
	}

	public EdgeKey ( String s )
	{
		parseString(s);
	}

	public EdgeKey( int bX, int bY, int eX, int eY, char tc, char bc, 
	                 double crp, double rob, int ct, String pk )
	{
		set( bX, bY, eX, eY, tc, bc );
		cumRp = crp;
		robustness = rob;
		count = ct;
		paramKey = pk;
	}

	private void init()
	{ 
		beginX = 0;
		beginY = 0;
		endX = 0;
		endY = 0;
		topChar = 'a';
		botChar = 'a';
		hashCode = 0;
		cumRp = 0.0;
		robustness = 0.0;
		count = 0;
		name = "";
		paramKey = "";
	}

	private void parseString( String s )
	{
		StringTokenizer st = new StringTokenizer( s, ".");

		char tc = st.nextToken().charAt(0);
		int bX = Integer.parseInt( st.nextToken() );
		int eX = Integer.parseInt( st.nextToken() );
		char bc = st.nextToken().charAt(0); 
		int bY = Integer.parseInt( st.nextToken() );
		int eY = Integer.parseInt( st.nextToken() );
			
		set( bX, bY, eX, eY, tc, bc );
	}

	public void set( int bX, int bY, int eX, int eY, char tc, char bc)
	{
		init();

		beginX = bX;
		beginY = bY;
		endX = eX;
		endY = eY;
		topChar = tc;
		botChar = bc;
		
		name = topChar + "." + Integer.toString(beginX) + "." +
			   Integer.toString(beginY) + "." + botChar + "." +
			   Integer.toString(endX) + "." + Integer.toString(endY);
				   
		hashCode = name.hashCode();
	}

	public int getBeginX()
	{ return beginX; }

	public int getBeginY()
	{ return beginY; }

	public int getEndX()
	{ return endX; }

	public int getEndY()
	{ return endY; }

	public char getTopChar()
	{ return topChar; }

	public char getBotChar()
	{ return botChar; }

	public String getTopCharString()
	{ return Character.toString( topChar ); }

	public String getBotCharString()
	{ return Character.toString( botChar ); }

	public boolean equals( Object ek )
	{
		if ( this == ek )
			return true;

		if ( ek instanceof EdgeKey )
		{
			if ( beginX == ((EdgeKey)ek).getBeginX() &&
				 beginY == ((EdgeKey)ek).getBeginY() &&
				 endX == ((EdgeKey)ek).getEndX() &&
				 endY == ((EdgeKey)ek).getEndY() &&
				 hashCode == ((EdgeKey)ek).hashCode() &&
				 topChar == ((EdgeKey)ek).getTopChar() &&
				 botChar == ((EdgeKey)ek).getBotChar() &&
				 name.equals(((EdgeKey)ek).getName()) //&&
				 //paramKey.equals(((EdgeKey)ek).getParamKey()) //&&
				 //robustness == ((EdgeKey)ek).getRobustness() &&
				 //count == ((EdgeKey)ek).getCount() && 
				 //cumRp == ((EdgeKey)ek).getCumulativePosition() 
				 )
				return true;
			else
				return false;
		}
		else
			return false;
	}

	public int hashCode()
	{ return hashCode; }

	public String toString()
	{
		String s = "EdgeKey: " + beginX + " " + beginY + " " + 
		           endX + " " + endY + " " + topChar + " " + 
				   botChar + " " + hashCode + "  " + count + " " +
				   robustness + " " + name + " " + paramKey;
		return s;
	}

	public double getCumulativePosition()
	{ return cumRp; }

	public int getCount()
	{ return count; }

	public void increment()
	{ count++; }

	public void decrement()
	{ count--; }

	public double getRobustness()
	{ return robustness; }

	public String getName()
	{ return name; }

	public String getParamKey()
	{ return paramKey; }

	public Object clone()
	{
		return new EdgeKey( beginX, beginY, endX, endY, topChar, botChar,
		                    cumRp, robustness, count, paramKey );
	}

    public void updateRp( double currRp )
    {
        // weighted avg.
        cumRp = (double)((count - 1)*cumRp + currRp) / (double)count;
    }
}

